#!/bin/bash

# 服务名称
SERVICE_NAME="sun"
SERVICE_FILE="/etc/systemd/system/sun.service"

# 颜色定义
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
PURPLE='\033[0;35m'
CYAN='\033[0;36m'
NC='\033[0m' # No Color

# 日志函数
log() {
    echo -e "${BLUE}[$(date '+%Y-%m-%d %H:%M:%S')]${NC} $1"
}

error_log() {
    echo -e "${RED}[$(date '+%Y-%m-%d %H:%M:%S')] ERROR:${NC} $1"
}

success_log() {
    echo -e "${GREEN}[$(date '+%Y-%m-%d %H:%M:%S')] SUCCESS:${NC} $1"
}

warning_log() {
    echo -e "${YELLOW}[$(date '+%Y-%m-%d %H:%M:%S')] WARNING:${NC} $1"
}

# 检查网络连接
check_network() {
    log "检查网络连接..."
    
    # 检查 DNS 解析
    if nslookup gz.zyfrp.vip &>/dev/null; then
        success_log "DNS 解析正常"
        return 0
    else
        warning_log "DNS 解析失败，尝试使用 IP 检查..."
    fi
    
    # 检查网络连通性
    local ping_targets=("114.114.114.114" "8.8.8.8" "1.1.1.1")
    for target in "${ping_targets[@]}"; do
        if ping -c 1 -W 3 $target &>/dev/null; then
            success_log "网络连通性正常 (通过 $target)"
            return 0
        fi
    done
    
    error_log "网络连接异常"
    return 1
}

# 等待网络就绪
wait_for_network() {
    local max_wait=60
    local wait_time=0
    
    log "等待网络就绪..."
    
    while [ $wait_time -lt $max_wait ]; do
        if check_network; then
            success_log "网络就绪"
            return 0
        fi
        
        warning_log "网络未就绪，等待 5 秒... ($((wait_time + 5))/$max_wait 秒)"
        sleep 5
        wait_time=$((wait_time + 5))
    done
    
    error_log "网络等待超时"
    return 1
}

# 检查服务文件配置
check_service_file() {
    log "检查服务文件配置..."
    
    if [ ! -f "$SERVICE_FILE" ]; then
        error_log "服务文件不存在: $SERVICE_FILE"
        return 1
    fi
    
    # 检查关键配置
    local has_errors=0
    
    if ! grep -q "\[Unit\]" "$SERVICE_FILE"; then
        error_log "服务文件缺少 [Unit] 段"
        has_errors=1
    fi
    
    if ! grep -q "\[Service\]" "$SERVICE_FILE"; then
        error_log "服务文件缺少 [Service] 段"
        has_errors=1
    fi
    
    if ! grep -q "\[Install\]" "$SERVICE_FILE"; then
        error_log "服务文件缺少 [Install] 段"
        has_errors=1
    fi
    
    # 检查 ExecStart 配置是否正确
    if ! grep -q "ExecStart=/frp/frpc" "$SERVICE_FILE"; then
        error_log "服务文件 ExecStart 配置错误（应该指向 /frp/frpc）"
        has_errors=1
    fi
    
    if ! grep -q "After=network.target" "$SERVICE_FILE" && ! grep -q "After=network-online.target" "$SERVICE_FILE"; then
        warning_log "服务文件缺少网络依赖配置"
    fi
    
    if grep -q "Unknown section" "$SERVICE_FILE"; then
        error_log "服务文件包含未知段"
        has_errors=1
    fi
    
    if [ $has_errors -eq 0 ]; then
        success_log "服务文件配置检查通过"
        return 0
    else
        error_log "服务文件配置检查失败"
        return 1
    fi
}

# 修复服务文件
fix_service_file() {
    log "修复服务文件..."
    
    # 备份原文件
    if [ -f "$SERVICE_FILE" ]; then
        cp "$SERVICE_FILE" "$SERVICE_FILE.backup.$(date +%Y%m%d_%H%M%S)"
        success_log "服务文件已备份: $SERVICE_FILE.backup.$(date +%Y%m%d_%H%M%S)"
    fi
    
    # 创建正确的服务文件
    cat > "$SERVICE_FILE" << 'EOF'
[Unit]
Description=FRP Client Service
After=network.target network-online.target
Wants=network-online.target

[Service]
Type=simple
User=root
WorkingDirectory=/frp
ExecStart=/frp/frpc -c /frp/frpc.ini
ExecStop=/bin/kill -TERM $MAINPID
ExecReload=/bin/kill -HUP $MAINPID
Restart=on-failure
RestartSec=10s
StartLimitInterval=60
StartLimitBurst=5
TimeoutStartSec=300

# 环境变量
Environment=FRP_LOG_LEVEL=info

[Install]
WantedBy=multi-user.target
EOF

    if [ $? -eq 0 ]; then
        success_log "服务文件修复完成"
        
        # 设置正确的权限
        chmod 644 "$SERVICE_FILE"
        success_log "服务文件权限设置完成"
        
        # 重新加载系统配置
        systemctl daemon-reload
        success_log "系统配置重新加载完成"
        
        return 0
    else
        error_log "服务文件修复失败"
        return 1
    fi
}

# 检查 FRP 相关文件
check_frp_files() {
    log "检查 FRP 相关文件..."
    
    local has_errors=0
    
    # 检查 frpc 二进制文件
    if [ ! -f "/frp/frpc" ]; then
        error_log "FRP 客户端二进制文件不存在: /frp/frpc"
        has_errors=1
    elif [ ! -x "/frp/frpc" ]; then
        warning_log "FRP 客户端二进制文件不可执行，尝试修复..."
        chmod +x /frp/frpc
        if [ $? -eq 0 ]; then
            success_log "FRP 客户端二进制文件权限修复完成"
        else
            error_log "FRP 客户端二进制文件权限修复失败"
            has_errors=1
        fi
    else
        success_log "FRP 客户端二进制文件检查通过"
    fi
    
    # 检查配置文件
    if [ ! -f "/frp/frpc.ini" ]; then
        error_log "FRP 配置文件不存在: /frp/frpc.ini"
        has_errors=1
    else
        success_log "FRP 配置文件检查通过"
        
        # 检查配置文件基本语法
        if grep -q "server_addr" "/frp/frpc.ini" && grep -q "server_port" "/frp/frpc.ini"; then
            success_log "FRP 配置文件基本语法检查通过"
        else
            warning_log "FRP 配置文件可能缺少必要配置"
        fi
    fi
    
    # 检查工作目录
    if [ ! -d "/frp" ]; then
        error_log "FRP 工作目录不存在: /frp"
        has_errors=1
    else
        success_log "FRP 工作目录检查通过"
    fi
    
    return $has_errors
}

# 检查服务状态的函数
check_service_status() {
    local service=$1
    log "检查服务状态..."
    
    # 检查服务是否在运行
    if systemctl is-active --quiet $service; then
        success_log "服务正在运行"
        
        # 检查服务是否启用开机自启
        if systemctl is-enabled --quiet $service; then
            success_log "服务已设置开机自启"
        else
            warning_log "服务未设置开机自启"
        fi
        
        # 显示详细的进程信息
        log "进程信息:"
        systemctl status $service --no-pager -l | grep -A 5 "Main PID" | head -6
        
        # 显示服务运行时间
        local uptime=$(systemctl show -p ActiveEnterTimestamp $service | cut -d= -f2)
        if [ -n "$uptime" ]; then
            success_log "服务启动时间: $uptime"
        fi
        
        return 0
    else
        error_log "服务未运行"
        
        # 检查失败原因
        local exit_code=$(systemctl show -p ExecMainStatus $service | cut -d= -f2)
        if [ "$exit_code" != "" ] && [ "$exit_code" != "0" ]; then
            error_log "服务退出代码: $exit_code"
        fi
        
        # 显示错误日志
        warning_log "最后几条日志:"
        journalctl -u $service --no-pager -n 10 --since "1 minute ago" 2>/dev/null || \
        echo "暂无相关日志"
        
        return 1
    fi
}

# 健康检查函数
health_check() {
    local service=$1
    local max_retries=8
    local retry_interval=2
    
    log "执行健康检查..."
    
    for ((i=1; i<=$max_retries; i++)); do
        warning_log "检查尝试 $i/$max_retries..."
        
        if systemctl is-active --quiet $service; then
            # 服务在运行，进行更深入的检查
            local pid=$(systemctl show -p MainPID $service | cut -d= -f2)
            if [ "$pid" != "0" ] && ps -p $pid > /dev/null 2>&1; then
                success_log "健康检查通过 - 服务PID $pid 正常运行"
                
                # 额外等待确保服务稳定
                sleep 1
                
                # 再次确认服务状态
                if systemctl is-active --quiet $service; then
                    success_log "服务运行稳定"
                    return 0
                else
                    error_log "服务运行不稳定，再次检查时已停止"
                    return 1
                fi
            else
                error_log "服务进程不存在 (PID: $pid)"
            fi
        else
            error_log "服务未运行"
        fi
        
        if [ $i -lt $max_retries ]; then
            warning_log "等待 ${retry_interval} 秒后重试..."
            sleep $retry_interval
        fi
    done
    
    error_log "健康检查失败 - 服务可能未正常运行"
    
    # 显示详细错误信息
    warning_log "详细错误信息:"
    systemctl status $service --no-pager -l
    
    return 1
}

# 重启服务函数
restart_service() {
    local service=$1
    log "重启服务 $service ..."
    
    # 先检查服务当前状态
    if systemctl is-active --quiet $service; then
        warning_log "服务正在运行，执行重启..."
    else
        warning_log "服务未运行，执行启动..."
    fi
    
    # 执行重启命令
    if systemctl restart $service; then
        success_log "重启命令执行成功"
    else
        error_log "重启命令执行失败"
        return 1
    fi
    
    # 等待服务重启完成
    warning_log "等待服务重启完成..."
    sleep 3
    
    # 执行健康检查
    if health_check $service; then
        success_log "服务重启成功！"
        return 0
    else
        error_log "服务重启失败"
        
        # 尝试诊断问题
        warning_log "尝试诊断问题..."
        
        # 检查服务配置
        if systemctl cat $service >/dev/null 2>&1; then
            success_log "服务配置文件存在"
        else
            error_log "服务配置文件不存在或无效"
        fi
        
        # 显示最近错误日志
        warning_log "最近错误日志:"
        journalctl -u $service --since "5 minutes ago" --no-pager | grep -i "error\|fail\|exception" | tail -5
        
        return 1
    fi
}

# 启动服务函数（带健康检查）
start_service() {
    local service=$1
    log "启动服务 $service ..."
    
    # 执行启动命令
    if systemctl start $service; then
        success_log "启动命令执行成功"
    else
        error_log "启动命令执行失败"
        return 1
    fi
    
    # 等待服务启动
    warning_log "等待服务启动..."
    sleep 3
    
    # 执行健康检查
    if health_check $service; then
        success_log "服务启动成功！"
        return 0
    else
        error_log "服务启动失败"
        return 1
    fi
}

# 完整系统检查
system_check() {
    log "开始完整系统检查..."
    
    echo -e "${CYAN}=== 网络检查 ===${NC}"
    check_network
    
    echo -e "${CYAN}=== FRP 文件检查 ===${NC}"
    check_frp_files
    
    echo -e "${CYAN}=== 服务文件检查 ===${NC}"
    check_service_file
    
    echo -e "${CYAN}=== 服务状态检查 ===${NC}"
    check_service_status $SERVICE_NAME
    
    log "完整系统检查完成"
}

# 修复所有问题
fix_all_issues() {
    log "开始修复所有已知问题..."
    
    echo -e "${CYAN}=== 修复服务文件 ===${NC}"
    if ! check_service_file; then
        fix_service_file
    else
        success_log "服务文件正常，无需修复"
    fi
    
    echo -e "${CYAN}=== 修复文件权限 ===${NC}"
    check_frp_files
    
    echo -e "${CYAN}=== 重载系统配置 ===${NC}"
    systemctl daemon-reload
    systemctl reset-failed $SERVICE_NAME
    success_log "系统配置重载完成"
    
    echo -e "${CYAN}=== 设置开机自启 ===${NC}"
    systemctl enable $SERVICE_NAME
    success_log "开机自启设置完成"
    
    success_log "所有问题修复完成"
}

# 完整启动服务
full_start() {
    log "开始完整启动流程..."
    
    echo -e "${CYAN}=== 步骤 1: 等待网络就绪 ===${NC}"
    if ! wait_for_network; then
        error_log "网络等待失败，但将继续尝试启动..."
    fi
    
    echo -e "${CYAN}=== 步骤 2: 系统检查和修复 ===${NC}"
    system_check
    fix_all_issues
    
    echo -e "${CYAN}=== 步骤 3: 最终配置检查 ===${NC}"
    # 确保服务文件正确
    if ! check_service_file; then
        error_log "服务文件检查失败，强制修复..."
        fix_service_file
    fi
    
    echo -e "${CYAN}=== 步骤 4: 设置开机自启 ===${NC}"
    if systemctl enable $SERVICE_NAME; then
        success_log "开机自启设置成功"
    else
        error_log "开机自启设置失败"
        return 1
    fi
    
    echo -e "${CYAN}=== 步骤 5: 等待网络完全就绪 ===${NC}"
    sleep 5
    
    echo -e "${CYAN}=== 步骤 6: 启动服务 ===${NC}"
    if start_service $SERVICE_NAME; then
        success_log "完整启动流程完成！服务运行正常"
        
        # 显示最终状态
        echo -e "${CYAN}=== 最终服务状态 ===${NC}"
        check_service_status $SERVICE_NAME
        
        # 显示连接信息
        echo -e "${CYAN}=== 连接信息 ===${NC}"
        log "服务启动完成，FRP 客户端应该正在运行"
        log "检查 FRP 连接状态..."
        sleep 2
        systemctl status $SERVICE_NAME --no-pager -l
    else
        error_log "启动流程完成，但服务可能未正常运行"
        warning_log "请检查服务日志：journalctl -u $SERVICE_NAME -f"
        
        # 显示详细的错误信息
        echo -e "${CYAN}=== 错误诊断 ===${NC}"
        systemctl status $SERVICE_NAME --no-pager -l
        return 1
    fi
}

# 显示菜单函数
show_menu() {
    clear
    echo -e "${BLUE}================================${NC}"
    echo -e "${BLUE}    $SERVICE_NAME 服务管理工具   ${NC}"
    echo -e "${BLUE}================================${NC}"
    echo -e "1.  ${GREEN}完整启动服务${NC} (推荐-带检测修复)"
    echo -e "2.  ${CYAN}系统状态检查${NC} (全面诊断)"
    echo -e "3.  ${YELLOW}修复所有问题${NC} (自动修复)"
    echo -e "4.  ${GREEN}启动服务${NC} (带健康检查)"
    echo -e "5.  ${RED}停止服务${NC}"
    echo -e "6.  ${BLUE}重启服务${NC} (带健康检查)"
    echo -e "7.  ${YELLOW}查看服务状态${NC} (详细)"
    echo -e "8.  ${PURPLE}查看服务日志${NC}"
    echo -e "9.  ${CYAN}健康检查${NC}"
    echo -e "10. ${GREEN}设置开机自启${NC}"
    echo -e "11. ${RED}关闭开机自启${NC}"
    echo -e "12. ${YELLOW}刷新服务列表${NC}"
    echo -e "13. ${RED}强制重载配置${NC}"
    echo -e "14. ${RED}退出脚本${NC}"
    echo -e "${BLUE}================================${NC}"
    echo -n "请选择操作 [1-14]: "
}

# 主程序
main() {
    # 检查是否以 root 权限运行
    if [ "$EUID" -ne 0 ]; then
        error_log "请使用 root 权限运行此脚本 (sudo $0)"
        exit 1
    fi
    
    # 创建日志目录
    local log_dir="/var/log/$SERVICE_NAME"
    mkdir -p "$log_dir"
    
    log "服务管理工具启动"
    log "服务名称: $SERVICE_NAME"
    log "服务文件: $SERVICE_FILE"
    
    # 显示当前状态
    if systemctl is-active --quiet $SERVICE_NAME; then
        success_log "服务当前状态: 运行中"
    else
        warning_log "服务当前状态: 未运行"
    fi
    
    while true; do
        show_menu
        read choice
        
        case $choice in
            1)
                echo -e "\n${GREEN}执行完整启动服务...${NC}"
                full_start
                ;;
            2)
                echo -e "\n${CYAN}执行系统状态检查...${NC}"
                system_check
                ;;
            3)
                echo -e "\n${YELLOW}修复所有问题...${NC}"
                fix_all_issues
                ;;
            4)
                echo -e "\n${GREEN}启动服务...${NC}"
                start_service $SERVICE_NAME
                ;;
            5)
                echo -e "\n${RED}停止服务...${NC}"
                systemctl stop $SERVICE_NAME
                success_log "服务已停止"
                ;;
            6)
                echo -e "\n${BLUE}重启服务...${NC}"
                restart_service $SERVICE_NAME
                ;;
            7)
                echo -e "\n${YELLOW}查看服务状态...${NC}"
                check_service_status $SERVICE_NAME
                ;;
            8)
                echo -e "\n${PURPLE}查看服务日志...${NC}"
                journalctl -u $SERVICE_NAME -n 20 --no-pager
                ;;
            9)
                echo -e "\n${CYAN}执行健康检查...${NC}"
                health_check $SERVICE_NAME
                ;;
            10)
                echo -e "\n${GREEN}设置开机自启...${NC}"
                systemctl enable $SERVICE_NAME
                success_log "开机自启设置完成"
                ;;
            11)
                echo -e "\n${RED}关闭开机自启...${NC}"
                systemctl disable $SERVICE_NAME
                success_log "开机自启已关闭"
                ;;
            12)
                echo -e "\n${YELLOW}刷新服务列表...${NC}"
                systemctl daemon-reload
                success_log "服务列表刷新完成"
                ;;
            13)
                echo -e "\n${RED}强制重载配置...${NC}"
                systemctl daemon-reload
                systemctl reset-failed $SERVICE_NAME
                success_log "配置重载完成"
                ;;
            14)
                echo -e "\n${BLUE}感谢使用，再见！${NC}"
                log "服务管理工具退出"
                exit 0
                ;;
            *)
                echo -e "\n${RED}无效选择，请重新输入！${NC}"
                ;;
        esac
        
        echo -e "\n按回车键继续..."
        read
    done
}

# 运行主程序
main "$@"