#!/bin/bash

# 服务名称
SERVICE_NAME="sun"

# 颜色定义
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
PURPLE='\033[0;35m'
CYAN='\033[0;36m'
NC='\033[0m' # No Color

# 检查服务状态的函数
check_service_status() {
    local service=$1
    echo -e "${CYAN}检查服务状态...${NC}"
    
    # 检查服务是否在运行
    if systemctl is-active --quiet $service; then
        echo -e "${GREEN}✓ 服务正在运行${NC}"
        
        # 检查服务是否启用开机自启
        if systemctl is-enabled --quiet $service; then
            echo -e "${GREEN}✓ 服务已设置开机自启${NC}"
        else
            echo -e "${YELLOW}⚠ 服务未设置开机自启${NC}"
        fi
        
        # 显示详细的进程信息
        echo -e "${CYAN}进程信息:${NC}"
        systemctl status $service --no-pager -l | grep -A 5 "Main PID" | head -6
        
        return 0
    else
        echo -e "${RED}✗ 服务未运行${NC}"
        
        # 检查失败原因
        local exit_code=$(systemctl show -p ExecMainStatus $service | cut -d= -f2)
        if [ "$exit_code" != "" ] && [ "$exit_code" != "0" ]; then
            echo -e "${RED}✗ 服务退出代码: $exit_code${NC}"
        fi
        
        # 显示错误日志
        echo -e "${YELLOW}最后几条日志:${NC}"
        journalctl -u $service --no-pager -n 10 --since "1 minute ago" 2>/dev/null || \
        echo "暂无相关日志"
        
        return 1
    fi
}

# 健康检查函数
health_check() {
    local service=$1
    local max_retries=8
    local retry_interval=2
    
    echo -e "${CYAN}执行健康检查...${NC}"
    
    for ((i=1; i<=$max_retries; i++)); do
        echo -e "${YELLOW}检查尝试 $i/$max_retries...${NC}"
        
        if systemctl is-active --quiet $service; then
            # 服务在运行，进行更深入的检查
            local pid=$(systemctl show -p MainPID $service | cut -d= -f2)
            if [ "$pid" != "0" ] && ps -p $pid > /dev/null 2>&1; then
                echo -e "${GREEN}✓ 健康检查通过 - 服务PID $pid 正常运行${NC}"
                
                # 额外等待确保服务稳定
                sleep 1
                
                # 再次确认服务状态
                if systemctl is-active --quiet $service; then
                    echo -e "${GREEN}✓ 服务运行稳定${NC}"
                    return 0
                else
                    echo -e "${RED}✗ 服务运行不稳定，再次检查时已停止${NC}"
                    return 1
                fi
            else
                echo -e "${RED}✗ 服务进程不存在 (PID: $pid)${NC}"
            fi
        else
            echo -e "${RED}✗ 服务未运行${NC}"
        fi
        
        if [ $i -lt $max_retries ]; then
            echo -e "${YELLOW}等待 ${retry_interval} 秒后重试...${NC}"
            sleep $retry_interval
        fi
    done
    
    echo -e "${RED}✗ 健康检查失败 - 服务可能未正常运行${NC}"
    
    # 显示详细错误信息
    echo -e "${YELLOW}详细错误信息:${NC}"
    systemctl status $service --no-pager -l
    
    return 1
}

# 重启服务函数
restart_service() {
    local service=$1
    echo -e "${BLUE}重启服务 $service ...${NC}"
    
    # 先检查服务当前状态
    if systemctl is-active --quiet $service; then
        echo -e "${YELLOW}服务正在运行，执行重启...${NC}"
    else
        echo -e "${YELLOW}服务未运行，执行启动...${NC}"
    fi
    
    # 执行重启命令
    if systemctl restart $service; then
        echo -e "${GREEN}✓ 重启命令执行成功${NC}"
    else
        echo -e "${RED}✗ 重启命令执行失败${NC}"
        return 1
    fi
    
    # 等待服务重启完成
    echo -e "${YELLOW}等待服务重启完成...${NC}"
    sleep 3
    
    # 执行健康检查
    if health_check $service; then
        echo -e "${GREEN}🎉 服务重启成功！${NC}"
        return 0
    else
        echo -e "${RED}❌ 服务重启失败${NC}"
        
        # 尝试诊断问题
        echo -e "${YELLOW}尝试诊断问题...${NC}"
        
        # 检查服务配置
        if systemctl cat $service >/dev/null 2>&1; then
            echo -e "${GREEN}✓ 服务配置文件存在${NC}"
        else
            echo -e "${RED}✗ 服务配置文件不存在或无效${NC}"
        fi
        
        # 显示最近错误日志
        echo -e "${YELLOW}最近错误日志:${NC}"
        journalctl -u $service --since "5 minutes ago" --no-pager | grep -i "error\|fail\|exception" | tail -5
        
        return 1
    fi
}

# 启动服务函数（带健康检查）
start_service() {
    local service=$1
    echo -e "${GREEN}启动服务 $service ...${NC}"
    
    # 执行启动命令
    if systemctl start $service; then
        echo -e "${GREEN}✓ 启动命令执行成功${NC}"
    else
        echo -e "${RED}✗ 启动命令执行失败${NC}"
        return 1
    fi
    
    # 等待服务启动
    echo -e "${YELLOW}等待服务启动...${NC}"
    sleep 3
    
    # 执行健康检查
    if health_check $service; then
        echo -e "${GREEN}🎉 服务启动成功！${NC}"
        return 0
    else
        echo -e "${RED}❌ 服务启动失败${NC}"
        return 1
    fi
}

# 完整启动服务
full_start() {
    echo -e "${YELLOW}第一步：刷新服务列表...${NC}"
    if systemctl daemon-reload; then
        echo -e "${GREEN}✓ 服务列表刷新成功${NC}"
    else
        echo -e "${RED}✗ 服务列表刷新失败${NC}"
        return 1
    fi
    
    echo -e "${YELLOW}第二步：设置开机自启...${NC}"
    if systemctl enable $SERVICE_NAME; then
        echo -e "${GREEN}✓ 开机自启设置成功${NC}"
    else
        echo -e "${RED}✗ 开机自启设置失败${NC}"
        return 1
    fi
    
    echo -e "${YELLOW}第三步：启动服务...${NC}"
    if start_service $SERVICE_NAME; then
        echo -e "${GREEN}🎉 完整启动流程完成！服务运行正常${NC}"
    else
        echo -e "${RED}❌ 启动流程完成，但服务可能未正常运行${NC}"
        echo -e "${YELLOW}请检查服务日志：journalctl -u $SERVICE_NAME -f${NC}"
        return 1
    fi
    
    # 显示最终状态
    check_service_status $SERVICE_NAME
}

# 显示菜单函数
show_menu() {
    clear
    echo -e "${BLUE}================================${NC}"
    echo -e "${BLUE}    $SERVICE_NAME 服务管理工具   ${NC}"
    echo -e "${BLUE}================================${NC}"
    echo -e "1. ${GREEN}完整启动服务${NC} (刷新+启用+启动+健康检查)"
    echo -e "2. ${YELLOW}刷新服务列表${NC}"
    echo -e "3. ${GREEN}设置开机自启${NC}"
    echo -e "4. ${RED}关闭开机自启${NC}"
    echo -e "5. ${GREEN}启动服务${NC} (带健康检查)"
    echo -e "6. ${RED}停止服务${NC}"
    echo -e "7. ${YELLOW}查看服务状态${NC} (详细)"
    echo -e "8. ${BLUE}重启服务${NC} (带健康检查)"
    echo -e "9. ${PURPLE}查看服务日志${NC}"
    echo -e "10. ${CYAN}健康检查${NC}"
    echo -e "11. ${RED}强制重载配置${NC}"
    echo -e "12. ${RED}退出脚本${NC}"
    echo -e "${BLUE}================================${NC}"
    echo -n "请选择操作 [1-12]: "
}

# 主程序
while true; do
    show_menu
    read choice
    
    case $choice in
        1)
            echo -e "\n${GREEN}执行完整启动服务...${NC}"
            full_start
            ;;
        2)
            echo -e "\n${YELLOW}刷新服务列表...${NC}"
            systemctl daemon-reload
            echo -e "${GREEN}服务列表刷新完成${NC}"
            ;;
        3)
            echo -e "\n${GREEN}设置开机自启...${NC}"
            systemctl enable $SERVICE_NAME
            echo -e "${GREEN}开机自启设置完成${NC}"
            ;;
        4)
            echo -e "\n${RED}关闭开机自启...${NC}"
            systemctl disable $SERVICE_NAME
            echo -e "${GREEN}开机自启已关闭${NC}"
            ;;
        5)
            echo -e "\n${GREEN}启动服务...${NC}"
            start_service $SERVICE_NAME
            ;;
        6)
            echo -e "\n${RED}停止服务...${NC}"
            systemctl stop $SERVICE_NAME
            echo -e "${GREEN}服务已停止${NC}"
            ;;
        7)
            echo -e "\n${YELLOW}查看服务状态...${NC}"
            check_service_status $SERVICE_NAME
            ;;
        8)
            echo -e "\n${BLUE}重启服务...${NC}"
            restart_service $SERVICE_NAME
            ;;
        9)
            echo -e "\n${PURPLE}查看服务日志...${NC}"
            journalctl -u $SERVICE_NAME -n 20 --no-pager
            ;;
        10)
            echo -e "\n${CYAN}执行健康检查...${NC}"
            health_check $SERVICE_NAME
            ;;
        11)
            echo -e "\n${RED}强制重载配置...${NC}"
            systemctl daemon-reload
            systemctl reset-failed $SERVICE_NAME
            echo -e "${GREEN}配置重载完成${NC}"
            ;;
        12)
            echo -e "\n${BLUE}感谢使用，再见！${NC}"
            exit 0
            ;;
        *)
            echo -e "\n${RED}无效选择，请重新输入！${NC}"
            ;;
    esac
    
    echo -e "\n按回车键继续..."
    read
done